/*******************************************************************************
* Copyright (C) 2019 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*
*  Content:
*       This example demonstrates use of oneAPI Math Kernel Library (oneMKL)
*       DPCPP API oneapi::mkl::sparse::gemvdot to perform general
*       sparse matrix-vector  multiplication on a SYCL device (CPU, GPU). This
*       example uses a sparse matrix in CSR format.
*
*       y = alpha * op(A) * x + beta * y
*       d = x * y
*
*       where op() is defined by one of
*           oneapi::mkl::transpose::{nontrans,trans,conjtrans}
*
*       The supported floating point data types for gemvdot are:
*           float
*           double
*           std::complex<float>
*           std::complex<double>
*
*       The supported matrix formats for gemvdot are:
*           CSR
*           COO (currently only on CPU device)
*
*******************************************************************************/

// stl includes
#include <algorithm>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <iterator>
#include <limits>
#include <list>
#include <vector>

#include "mkl.h"
#include "oneapi/mkl.hpp"
#include <sycl/sycl.hpp>

// local includes
#include "common_for_examples.hpp"
#include "./include/common_for_sparse_examples.hpp"

//
// Main example for Sparse Matrix-Vector Multiply Dot Product consisting of
// initialization of A matrix, x and y vectors as well as
// scalars alpha and beta.  Then the product
//
// y = alpha * op(A) * x + beta * y
// d = x * y
//
// is performed and finally the results are post processed.
//
template <typename dataType, typename intType>
int run_sparse_blas_example(sycl::queue &q)
{
    bool good = true;

    //
    // handle for sparse matrix
    //
    oneapi::mkl::sparse::matrix_handle_t csrA = nullptr;

    try {
        // Initialize data for Sparse Matrix-Vector Multiply Dot Product
        oneapi::mkl::transpose opA             = oneapi::mkl::transpose::nontrans;
        oneapi::mkl::index_base index_base_val = oneapi::mkl::index_base::zero;

        intType a_ind = (index_base_val == oneapi::mkl::index_base::zero ? 0 : 1);

        // Matrix data size
        intType size  = 4;
        const std::int64_t nrows = size * size * size;
        const std::int64_t ncols = nrows;

        // Input matrix in CSR format
        std::vector<intType, mkl_allocator<intType, 64>> ia;
        std::vector<intType, mkl_allocator<intType, 64>> ja;
        std::vector<dataType, mkl_allocator<dataType, 64>> a;

        ia.resize(nrows + 1);
        ja.resize(27 * nrows);
        a.resize(27 * nrows);

        generate_sparse_matrix<dataType, intType>(size, ia, ja, a, a_ind);
        const std::int64_t nnz = ia[nrows] - a_ind;

        // Vectors x and y
        std::vector<dataType, mkl_allocator<dataType, 64>> x;
        std::vector<dataType, mkl_allocator<dataType, 64>> y;
        std::vector<dataType, mkl_allocator<dataType, 64>> y_ref;
        std::vector<dataType, mkl_allocator<dataType, 64>> d;
        dataType d_ref = set_fp_value(dataType(0.0), dataType(0.0));

        x.resize(nrows);
        y.resize(nrows);
        y_ref.resize(nrows);
        d.resize(1);

        // Init vectors x and y
        for (int i = 0; i < nrows; i++) {
            x[i] = set_fp_value(dataType(1.0), dataType(0.0));
            y[i] = set_fp_value(dataType(0.0), dataType(0.0));
            y_ref[i] = set_fp_value(dataType(0.0), dataType(0.0));
        }

        // Set scalar dataType values
        dataType alpha = set_fp_value(dataType(1.0), dataType(0.0));
        dataType beta  = set_fp_value(dataType(0.0), dataType(0.0));

        //
        // Execute Matrix -Vector Multiply
        //

        std::cout << "\n\t\tsparse::gemvdot parameters:\n";
        std::cout << "\t\t\topA    = " << opA << std::endl;
        std::cout << "\t\t\tnrows  = " << nrows << std::endl;
        std::cout << "\t\t\talpha  = " << alpha << std::endl;
        std::cout << "\t\t\tbeta   = " << beta << std::endl;

        sycl::buffer<intType, 1> ia_buffer(ia.data(), ia.data() + nrows + 1 );
        sycl::buffer<intType, 1> ja_buffer(ja.data(), ja.data() + nnz);
        sycl::buffer<dataType, 1> a_buffer(a.data(), a.data() + nnz);
        sycl::buffer<dataType, 1> x_buffer(x.data(), x.data() + x.size());
        sycl::buffer<dataType, 1> y_buffer(y.data(), y.data() + y.size());
        sycl::buffer<dataType, 1> d_buffer(d.data(), d.data() + d.size());

        oneapi::mkl::sparse::init_matrix_handle(&csrA);

        oneapi::mkl::sparse::set_csr_data(q, csrA, nrows, ncols, nnz, index_base_val,
                                          ia_buffer, ja_buffer, a_buffer);

        // add oneapi::mkl::sparse::gemvdot to execution queue
        oneapi::mkl::sparse::gemvdot(q, opA, alpha, csrA,
                                     x_buffer, beta, y_buffer, d_buffer);

        oneapi::mkl::sparse::release_matrix_handle(q, &csrA);

        q.wait();

        //
        // Post Processing
        //

        // calculate y_ref = alpha op(A) * x + beta * y_ref
        auto res = d_buffer.get_host_access(sycl::read_only);
        if (opA == oneapi::mkl::transpose::nontrans) {
            for (intType row = 0; row < nrows; row++) {
                dataType tmp = set_fp_value(dataType(0.0), dataType(0.0));
                for (intType i = ia[row] - a_ind; i < ia[row + 1] - a_ind; i++) {
                    const intType col = ja[i] - a_ind;
                    tmp += a[i] * x[col];
                }
                y_ref[row] = alpha * tmp + beta * y_ref[row];
            }
        }
        else { // opA is trans or conjtrans
            const bool isConj = (opA == oneapi::mkl::transpose::conjtrans);
            for (intType row = 0; row < nrows; row++) {
                y_ref[row] *= beta;
            }
            for (intType row = 0; row < nrows; row++) {
                dataType tmp = alpha * x[row];
                for (intType i = ia[row]-a_ind; i < ia[row+1]-a_ind; i++) {
                    if constexpr (is_complex<dataType>()) {
                        y_ref[ja[i]-a_ind] += tmp * (isConj ? std::conj(a[i]) : a[i]);
                    }
                    else {
                        y_ref[ja[i]-a_ind] += tmp * a[i];
                    }
                }
            }
        }

        // calculate d = dot(x, y_ref);
        for (intType i = 0; i < nrows; i++) {
            d_ref += x[i] * y_ref[i];
        }

        dataType diff = set_fp_value(dataType(0.0), dataType(0.0));
        diff = d_ref - res[0];

        std::cout << "\n\t\t sparse::gemvdot residual:\n"
                  << "\t\t\t" << std::abs(diff) << "\n\tFinished" << std::endl;

        if (std::abs(diff) > 0) { good = false; }

        q.wait_and_throw();

    }
    catch (sycl::exception const &e) {
        std::cout << "\t\tCaught synchronous SYCL exception:\n" << e.what() << std::endl;
        good = false;
    }
    catch (std::exception const &e) {
        std::cout << "\t\tCaught std exception:\n" << e.what() << std::endl;
        good = false;
    }

    q.wait();

    // backup cleaning of matrix handle and others for if exceptions happened
    if (csrA) oneapi::mkl::sparse::release_matrix_handle(q, &csrA);

    q.wait();

    return good ? 0 : 1;
}

//
// Description of example setup, apis used and supported floating point type
// precisions
//
void print_example_banner()
{

    std::cout << "" << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << "# Sparse Matrix-Vector Multiply Dot Product Example: " << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# y = alpha * op(A) * x + beta * y" << std::endl;
    std::cout << "# d = x * y" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# where A is a sparse matrix in CSR format, x and y are "
                 "dense vectors"
              << std::endl;
    std::cout << "# and alpha, beta, d are floating point type precision scalars." << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Using apis:" << std::endl;
    std::cout << "#   sparse::gemvdot" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "# Supported floating point type precisions:" << std::endl;
    std::cout << "#   float" << std::endl;
    std::cout << "#   double" << std::endl;
    std::cout << "#   std::complex<float>" << std::endl;
    std::cout << "#   std::complex<double>" << std::endl;
    std::cout << "# " << std::endl;
    std::cout << "###############################################################"
                 "#########"
              << std::endl;
    std::cout << std::endl;
}

//
// Main entry point for example.
//
// Dispatches to appropriate device types as set at build time with flag:
// -DSYCL_DEVICES_cpu -- only runs SYCL CPU implementation
// -DSYCL_DEVICES_gpu -- only runs SYCL GPU implementation
// -DSYCL_DEVICES_all (default) -- runs on all: cpu and gpu devices
//
//  For each device selected and each supported data type,
//  run_sparse_blas_example is run with all supported data types,
//  if any fail, we move on to the next device.
//

int main(int argc, char **argv)
{

    print_example_banner();

    std::list<my_sycl_device_types> list_of_devices;
    set_list_of_devices(list_of_devices);

    int status = 0;
    for (auto it = list_of_devices.begin(); it != list_of_devices.end(); ++it) {
        try {
            sycl::device my_dev;
            bool my_dev_is_found = false;
            get_sycl_device(my_dev, my_dev_is_found, *it);

            if (my_dev_is_found) {
                std::cout << "Running tests on " << sycl_device_names[*it] << ".\n";

                // Catch asynchronous exceptions
                auto exception_handler = [](sycl::exception_list exceptions) {
                    for (std::exception_ptr const &e : exceptions) {
                        try {
                            std::rethrow_exception(e);
                        }
                        catch (sycl::exception const &e) {
                            std::cout << "Caught asynchronous SYCL exception: \n"
                                << e.what() << std::endl;
                        }
                    }
                };

                sycl::queue q(my_dev, exception_handler);

                std::cout << "\tRunning with single precision real data type:" << std::endl;
                status |= run_sparse_blas_example<float, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision real data type:" << std::endl;
                    status |= run_sparse_blas_example<double, std::int32_t>(q);
                }

                std::cout << "\tRunning with single precision complex data type:" << std::endl;
                status |= run_sparse_blas_example<std::complex<float>, std::int32_t>(q);

                if (my_dev.get_info<sycl::info::device::double_fp_config>().size() != 0) {
                    std::cout << "\tRunning with double precision complex data type:" << std::endl;
                    status |= run_sparse_blas_example<std::complex<double>, std::int32_t>(q);
                }

            }
            else {
#ifdef FAIL_ON_MISSING_DEVICES
                std::cout << "No " << sycl_device_names[*it]
                    << " devices found; Fail on missing devices "
                    "is enabled.\n";
                return 1;
#else
                std::cout << "No " << sycl_device_names[*it] << " devices found; skipping "
                    << sycl_device_names[*it] << " tests.\n";
#endif
            }
        }
        catch (sycl::exception const &e) {
            std::cout << "\t\tCaught SYCL exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }
        catch (std::exception const &e) {
            std::cout << "\t\tCaught std exception at driver level: \n" << e.what() << std::endl;
            continue; // stop with device, but move on to other devices
        }


    } // for loop over devices

    mkl_free_buffers();
    return status;
}
