//=================================================================================================
/*!
//  \file blaze/util/typetraits/IsValid.h
//  \brief Header file for the IsValid type trait
//
//  Copyright (C) 2012-2020 Klaus Iglberger - All Rights Reserved
//
//  This file is part of the Blaze library. You can redistribute it and/or modify it under
//  the terms of the New (Revised) BSD License. Redistribution and use in source and binary
//  forms, with or without modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright notice, this list of
//     conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright notice, this list
//     of conditions and the following disclaimer in the documentation and/or other materials
//     provided with the distribution.
//  3. Neither the names of the Blaze development group nor the names of its contributors
//     may be used to endorse or promote products derived from this software without specific
//     prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
//  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
//  SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
//  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
//  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
//  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
*/
//=================================================================================================

#ifndef _BLAZE_UTIL_TYPETRAITS_ISVALID_H_
#define _BLAZE_UTIL_TYPETRAITS_ISVALID_H_


//*************************************************************************************************
// Includes
//*************************************************************************************************

#include <blaze/util/IntegralConstant.h>
#include <blaze/util/InvalidType.h>


namespace blaze {

//=================================================================================================
//
//  CLASS DEFINITION
//
//=================================================================================================

//*************************************************************************************************
/*!\brief Compile time type check.
// \ingroup type_traits
//
// This class tests whether the given template parameter is a valid or invalid data type (i.e.
// if the type is the INVALID_TYPE). If \a T is not the INVALID_TYPE class type, the \a value
// member constant is set to \a true, the nested type definition \a Type is \a TrueType, and
// the class derives from \a TrueType. Otherwise \a value is set to \a false, \a Type is
// \a FalseType, and the class derives from \a FalseType.

   \code
   blaze::IsValid<int>::value                // Evaluates to 'true'
   blaze::IsValid<float const>::Type         // Results in TrueType
   blaze::IsValid<double volatile>           // Is derived from TrueType
   blaze::IsValid<INVALID_TYPE>::value       // Evaluates to 'false'
   blaze::IsValid<INVALID_TYPE const>::Type  // Results in FalseType
   blaze::IsValid<INVALID_TYPE volatile>     // Is derived from FalseType
   \endcode
*/
template< typename T >
struct IsValid
   : public TrueType
{};
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
//! Specialization of the IsValid type trait for the plain 'INVALID_TYPE' type.
template<>
struct IsValid<INVALID_TYPE>
   : public FalseType
{};
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
//! Specialization of the IsValid type trait for 'const INVALID_TYPE'.
template<>
struct IsValid<const INVALID_TYPE>
   : public FalseType
{};
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
//! Specialization of the IsValid type trait for 'volatile INVALID_TYPE'.
template<>
struct IsValid<volatile INVALID_TYPE>
   : public FalseType
{};
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*! \cond BLAZE_INTERNAL */
//! Specialization of the IsValid type trait for 'const volatile INVALID_TYPE'.
template<>
struct IsValid<const volatile INVALID_TYPE>
   : public FalseType
{};
/*! \endcond */
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Auxiliary variable template for the IsValid type trait.
// \ingroup type_traits
//
// The IsValid_v variable template provides a convenient shortcut to access the nested \a value
// of the IsValid class template. For instance, given the type \a T the following two statements
// are identical:

   \code
   constexpr bool value1 = blaze::IsValid<T>::value;
   constexpr bool value2 = blaze::IsValid_v<T>;
   \endcode
*/
template< typename T >
constexpr bool IsValid_v = IsValid<T>::value;
//*************************************************************************************************

} // namespace blaze

#endif
