#!/bin/bash

#===============================================================================
# Copyright 2019-2023 Intel Corporation
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#===============================================================================

# Get absolute path to script, when sourced from bash, zsh and ksh shells.
# Uses `readlink` to remove links and `pwd -P` to turn into an absolute path.
# Derived from a function created by Sergey Shkrunin for use by the analyzers.
# Converted into a POSIX-compliant function.

# Usage:
#   script_dir=$(get_script_path "$script_rel_path")
#
# Inputs:
#   script/relative/pathname/scriptname
#
# Outputs:
#   /script/absolute/pathname

# executing function in a *subshell* to localize vars and effects on `cd`
get_script_path() (
  script="$1"
  while [ -L "$script" ] ; do
    # combining next two lines fails in zsh shell
    script_dir=$(command dirname -- "$script")
    script_dir=$(cd "$script_dir" && command pwd -P)
    script="$(readlink "$script")"
    case $script in
      (/*) ;;
       (*) script="$script_dir/$script" ;;
    esac
  done
  # combining next two lines fails in zsh shell
  script_dir=$(command dirname -- "$script")
  script_dir=$(cd "$script_dir" && command pwd -P)
  echo "$script_dir"
)

# Check to insure that this script is being sourced, not executed.

# This script is designed to be POSIX compatible, there are a few lines in the
# code block below that contain elements that are specific to a shell. The
# shell-specific elements are needed to identify the sourcing shell.

usage() {
  printf "%s\n"   "ERROR: This script must be sourced."
  printf "%s\n"   "Usage: source $1"
  exit 255
}

vars_script_name=""

vars_script_shell="$(ps -p "$$" -o comm=)"
if [ -n "${ZSH_VERSION:-}" ] && [ -n "${ZSH_EVAL_CONTEXT:-}" ] ; then     # zsh 5.x and later
  case $ZSH_EVAL_CONTEXT in (*:file*) vars_script_name="${(%):-%x}" ;; esac ;
elif [ -n "${KSH_VERSION:-}" ] ; then                                     # ksh, mksh or lksh
  if [ "$(set | grep -Fq "KSH_VERSION=.sh.version" ; echo $?)" -eq 0 ] ; then # ksh
    vars_script_name="${.sh.file}" ;
  else # mksh or lksh or [lm]ksh masquerading as ksh or sh
    # force [lm]ksh to issue error msg; which contains this script's path/filename, e.g.:
    # mksh: /home/ubuntu/intel/oneapi/vars.sh[137]: ${.sh.file}: bad substitution
    vars_script_name="$( (echo "${.sh.file}") 2>&1 )" || : ;
    vars_script_name="$(expr "${vars_script_name:-}" : '^.*sh: \(.*\)\[[0-9]*\]:')" ;
  fi
elif [ -n "${BASH_VERSION:-}" ] ; then        # bash
  (return 0 2>/dev/null) && vars_script_name="${BASH_SOURCE}" ;
elif [ "dash" = "$vars_script_shell" ] ; then # dash
  # force dash to issue error msg; which contains this script's rel/path/filename, e.g.:
  # dash: 146: /home/ubuntu/intel/oneapi/vars.sh: Bad substitution
  vars_script_name="$( (echo "${.sh.file}") 2>&1 )" || : ;
  vars_script_name="$(expr "${vars_script_name:-}" : '^.*dash: [0-9]*: \(.*\):')" ;
elif [ "sh" = "$vars_script_shell" ] ; then   # could be dash masquerading as /bin/sh
  # force a shell error msg; which should contain this script's path/filename
  # sample error msg shown; assume this file is named "vars.sh"; as required by setvars.sh
  vars_script_name="$( (echo "${.sh.file}") 2>&1 )" || : ;
  if [ "$(printf "%s" "$vars_script_name" | grep -Eq "sh: [0-9]+: .*vars\.sh: " ; echo $?)" -eq 0 ] ; then # dash as sh
    # sh: 155: /home/ubuntu/intel/oneapi/vars.sh: Bad substitution
    vars_script_name="$(expr "${vars_script_name:-}" : '^.*sh: [0-9]*: \(.*\):')" ;
  fi
else  # unrecognized shell or dash being sourced from within a user's script
  # force a shell error msg; which should contain this script's path/filename
  # sample error msg shown; assume this file is named "vars.sh"; as required by setvars.sh
  vars_script_name="$( (echo "${.sh.file}") 2>&1 )" || : ;
  if [ "$(printf "%s" "$vars_script_name" | grep -Eq "^.+: [0-9]+: .*vars\.sh: " ; echo $?)" -eq 0 ] ; then # dash
    # .*: 164: intel/oneapi/vars.sh: Bad substitution
    vars_script_name="$(expr "${vars_script_name:-}" : '^.*: [0-9]*: \(.*\):')" ;
  else
    vars_script_name="" ;
  fi
fi

if [ "" = "$vars_script_name" ] ; then
  >&2 echo "   ERROR: Unable to proceed: possible causes listed below."
  >&2 echo "   This script must be sourced. Did you execute or source this script?" ;
  >&2 echo "   Unrecognized/unsupported shell (supported: bash, zsh, ksh, m/lksh, dash)." ;
  >&2 echo "   May fail in dash if you rename this script (assumes \"vars.sh\")." ;
  >&2 echo "   Can be caused by sourcing from ZSH version 4.x or older." ;
  return 255 2>/dev/null || exit 255
fi

__dnnl_this_dir_path=$(get_script_path "$vars_script_name")

dnnl_set_env_vars() {
  export DNNLROOT="$1"

  if [ -n "${LD_LIBRARY_PATH:-}" ] ; then
      export LD_LIBRARY_PATH=$DNNLROOT/lib:$LD_LIBRARY_PATH
  else
      export LD_LIBRARY_PATH=$DNNLROOT/lib
  fi

  if [ -n "${LIBRARY_PATH:-}" ] ; then
      export LIBRARY_PATH=$DNNLROOT/lib:$LIBRARY_PATH
  else
      export LIBRARY_PATH=$DNNLROOT/lib
  fi

  if [ -n "${C_INCLUDE_PATH:-}" ] ; then
      export C_INCLUDE_PATH=$DNNLROOT/include:$C_INCLUDE_PATH
  else
      export C_INCLUDE_PATH=$DNNLROOT/include
  fi

  if [ -n "${CPLUS_INCLUDE_PATH:-}" ] ; then
      export CPLUS_INCLUDE_PATH=$DNNLROOT/include:$CPLUS_INCLUDE_PATH
  else
      export CPLUS_INCLUDE_PATH=$DNNLROOT/include
  fi

  if [ -n "${PKG_CONFIG_PATH:-}" ] ; then
      export PKG_CONFIG_PATH=$DNNLROOT/lib/pkgconfig:$PKG_CONFIG_PATH
  else
      export PKG_CONFIG_PATH=$DNNLROOT/lib/pkgconfig
  fi

  if [ -n "${CMAKE_PREFIX_PATH:-}" ] ; then
      export CMAKE_PREFIX_PATH=$DNNLROOT/lib/cmake:$CMAKE_PREFIX_PATH
  else
      export CMAKE_PREFIX_PATH=$DNNLROOT/lib/cmake
  fi
}

dnnl_env_vars_main() {
  __dnnl_root="$(dirname "$__dnnl_this_dir_path")"
  dnnl_set_env_vars "$__dnnl_root"
}

dnnl_env_vars_main

unset __dnnl_this_dir_path
